﻿"""Implementation of an orbit transport.

This transport orbits around a node in the scene specified by the targetNode
parameter.

The orbit transport currently does not make use of the pivot parameter. This is
due to the fact that for this method of transportation a pivot may be too confusing.
"""

import viz
import vizmat

import transportation


class Orbit(transportation.AccelerationTransport):
	"""A transport class used for orbiting a set object or location"""
	def __init__(self, targetNode=None,
				useParentOrientation=True,
				orbitDistance=2.0, # in m
				autoOrbitSpeedIncrement=2.0,
				acceleration=4.0, # in meters per second per second, lower accelerations can be obtained by using a smaller mag on the input, e.g. pressing the joystick lower
				maxSpeed=10.44, # in meters per second, as a reference 1.4m/s is a typical walking speed, 10.44 is a very fast run
				rotationAcceleration=90.0, # in degrees per second per second
				maxRotationSpeed=120.0, # in degrees per second
				autoBreakingDragCoef=0.1, # determines how quickly the walking transport will stop 
				dragCoef=0.0001,
				rotationAutoBreakingDragCoef=0.2, # determines how quickly the walking transport will stop 
				rotationDragCoef=0.0001, # normal drag coef
				updatePriority=29, # higher for orbit
				autoBreakingTimeout=0, # how long before auto breaking is enabled
				rotationAutoBreakingTimeout=0, # how long before rotational auto breaking is enabled
				**kwargs):
		
		super(Orbit, self).__init__(acceleration=acceleration,
									maxSpeed=maxSpeed,
									rotationAcceleration=rotationAcceleration,
									maxRotationSpeed=maxRotationSpeed,
									autoBreakingDragCoef=autoBreakingDragCoef,
									dragCoef=dragCoef,
									rotationAutoBreakingDragCoef=rotationAutoBreakingDragCoef,
									rotationDragCoef=rotationDragCoef,
									updatePriority=updatePriority,
									**kwargs)
		
		self.LOCK_INCREASE_ORBIT_LEFT = 1
		self.LOCK_INCREASE_ORBIT_RIGHT = 2
		
		self._yaw = 0
		self._pitch = 0
		self._roll = 0
		
		self._orbitSpeed = [0, 0, 0]
		self._targetNode = targetNode
		self._useParentOrientation = useParentOrientation
		self._orbitDistance = orbitDistance
		self._autoOrbitSpeedIncrement = autoOrbitSpeedIncrement
		self._rotationAutoBreakingTimeout = rotationAutoBreakingTimeout
		self._autoBreakingTimeout = autoBreakingTimeout
		
		self._autoBreakingTimeoutCounter = 0
		self._rotationAutoBreakingTimeoutCounter = [0, 0, 0]
		
		self.setUpdateFunction(lambda transport: None)
		
		self.finalize()
	
	def getDistance(self):
		"""Returns the distance of the orbiting object to the center of the orbit"""
		return self._orbitDistance
	
	def setDistance(self, dist):
		"""Sets the distance of the orbiting object to the center of the orbit"""
		self._orbitDistance = dist
		if not self._deferred:
			self.finalize()
	
	def zoomIn(self, mag=1):
		"""Zooms the orbit transport in."""
		self._Ap[2] = -pow(mag, self._exp)
		if not self._deferred:
			self.finalize()
	
	def zoomOut(self, mag=1):
		"""Zooms the orbit transport out."""
		self._Ap[2] = pow(mag, self._exp)
		if not self._deferred:
			self.finalize()
	
	def left(self, mag=1):
		"""Orbits left."""
		self._Ar[0] = pow(mag, self._exp)
		if not self._deferred:
			self.finalize()
	
	def right(self, mag=1):
		"""Orbits right."""
		self._Ar[0] = -pow(mag, self._exp)
		if not self._deferred:
			self.finalize()
	
	def rollLeft(self, mag=1):
		"""Rolls the orbit transport left."""
		self._Ar[2] = -pow(mag, self._exp)
		if not self._deferred:
			self.finalize()
	
	def rollRight(self, mag=1):
		"""Rolls the orbit transport right."""
		self._Ar[2] = pow(mag, self._exp)
		if not self._deferred:
			self.finalize()
	
	def up(self, mag=1):
		"""Pitches the orbit transport up."""
		self._Ar[1] = pow(mag, self._exp)
		if not self._deferred:
			self.finalize()
	
	def down(self, mag=1):
		"""Pitches the orbit transport down."""
		self._Ar[1] = -pow(mag, self._exp)
		if not self._deferred:
			self.finalize()
	
	def increaseOrbitLeft(self, mag=1):
		"""Increases the orbit transport's rotation left."""
		if not self._frameLocked&self.LOCK_INCREASE_ORBIT_LEFT:
			self._orbitSpeed[0] += self._autoOrbitSpeedIncrement
		self._lockRequested |= self.LOCK_INCREASE_ORBIT_LEFT
	
	def increaseOrbitRight(self, mag=1):
		"""Increases the orbit transport's rotation right."""
		if not self._frameLocked&self.LOCK_INCREASE_ORBIT_RIGHT:
			self._orbitSpeed[0] -= self._autoOrbitSpeedIncrement
		self._lockRequested |= self.LOCK_INCREASE_ORBIT_RIGHT
	
	def setOrbitSpeed(self, orbitSpeed):
		"""Sets the left/right orbit speed."""
		self._orbitSpeed = orbitSpeed[:]
	
	def finalize(self):
		"""Function which executes the quequed functions such as
		moveForward and moveBack basing them off the sample orientation
		from the tracker. Should be called regularly either by a timer
		or ideally at every frame.
		"""
		self._frameLocked = self._lockRequested
		self._lockRequested = 0
		
		self._updateTime()
		if self._dt == 0:
			self._dt = 0.0001
		idt = min(60.0, 1.0/self._dt)
		
		# if necessary normalize the acceleration
		mag = self._Ap.length()
		if mag > 1.0:
			self._Ap = self._Ap / mag
		# .. and for rotation
		mag = self._Ar.length()
		if mag > 1.0:
			self._Ar = self._Ar / mag
		
		# scale acceleration (right now no units just 0-1 range magnitude vector)
		self._Ap *= self._acceleration
		# .. and for rotation
		self._Ar *= self._rotationAcceleration
		
		# update velocity for position/zoom
		breaking = self._Vp[2] * self._autoBreakingDragCoef * idt
		if self._Ap[2] != 0 and (self._Ap[2]*self._Vp[2] > 0):
			breaking = 0# cancel breaking
		if breaking:
			if self._autoBreakingTimeoutCounter < self._autoBreakingTimeout:
				breaking = 0# cancel breaking
			self._autoBreakingTimeoutCounter += self._dt
		else:
			self._autoBreakingTimeoutCounter = 0
		
		drag = self._Vp[2] * self._dragCoef
		self._Vp[2] += (self._Ap[2] - drag - breaking)*self._dt
		velMag = abs(self._Vp[2])
		if velMag > self._maxSpeed:
			self._Vp[2] = (self._Vp[2] / velMag) * self._maxSpeed
		
		# .. and for rotation
		breakingVec = self._Vr * self._rotationAutoBreakingDragCoef * idt
		for i in range(0, 3):
			if self._Ar[i] != 0:
				breakingVec[i] = 0
			if breakingVec[i]:
				if self._rotationAutoBreakingTimeoutCounter[i] < self._rotationAutoBreakingTimeout:
					breakingVec[i] = 0# cancel breaking
				self._rotationAutoBreakingTimeoutCounter[i] += self._dt
			else:
				self._rotationAutoBreakingTimeoutCounter[i] = 0
		
		drag = self._Vr * self._rotationDragCoef
		self._Vr[0] += (self._Ar[0] - drag[0] - breakingVec[0]) * self._dt
		self._Vr[1] += (self._Ar[1] - drag[1] - breakingVec[1]) * self._dt
		self._Vr[2] += (self._Ar[2] - drag[2] - breakingVec[2]) * self._dt
		velMag = self._Vr.length()
		if velMag > self._maxRotationSpeed:
			self._Vr = (self._Vr / velMag) * self._maxRotationSpeed
		
		self._orbitDistance = min(1000, max(0.01, self._orbitDistance + self._Vp[2]*self._movementSpeed[2]*self._dt))
		
		self._yaw += self._Vr[0]*self._dt + self._orbitSpeed[0]*self._dt*self._movementSpeed[0]
		self._pitch += self._Vr[1]*self._dt + self._orbitSpeed[1]*self._dt*self._movementSpeed[1]
		self._roll += self._Vr[2]*self._dt + self._orbitSpeed[2]*self._dt
		
		parents = self.getParents()
		if parents:
			pos = parents[0].getPosition(viz.ABS_GLOBAL)
		else:
			pos = [0, 0, 0]
		
		originMat = vizmat.Transform()
		originMat.setEuler([self._yaw, -self._pitch, self._roll])
		
		up = originMat.preMultVec([0, 1, 0])
		eye = originMat.preMultVec([0, 0, self._orbitDistance])
		
		mat = vizmat.Transform()
		mat.makeLookAt(eye, [0, 0, 0], up)
		
		if self._useParentOrientation:
			self.setMatrix(mat)
		else:
			mat.setPosition(eye[0]+pos[0], eye[1]+pos[1], eye[2]+pos[2])
			self.setMatrix(mat, viz.ABS_GLOBAL)
		
		self._Ap[0] = 0
		self._Ap[1] = 0
		self._Ap[2] = 0
		self._Ar[0] = 0
		self._Ar[1] = 0
		self._Ar[2] = 0


if __name__ == "__main__":
	import vizact
	viz.go()
	
	# load a model
	piazza = viz.add('piazza.osgb')
	
	transportRepresentation = viz.add("beachball.osgb")
	orbit = Orbit(targetNode=None,
				useParentOrientation=True,
				orbitDistance=2.0, # in m
				autoOrbitSpeedIncrement=2.0,
				acceleration=4.0, # in meters per second per second, lower accelerations can be obtained by using a smaller mag on the input, e.g. pressing the joystick lower
				maxSpeed=10.44, # in meters per second, as a reference 1.4m/s is a typical walking speed, 10.44 is a very fast run
				rotationAcceleration=90.0, # in degrees per second per second
				maxRotationSpeed=120.0, # in degrees per second
				autoBreakingDragCoef=0.1, # determines how quickly the walking transport will stop 
				dragCoef=0.0001,
				rotationAutoBreakingDragCoef=0.2, # determines how quickly the walking transport will stop 
				rotationDragCoef=0.0001, # normal drag coef
				updatePriority=0,
				autoBreakingTimeout=0.1,
				rotationAutoBreakingTimeout=0.1) # higher for orbit
	
	vizact.onkeydown(viz.KEY_UP, orbit.up)
	vizact.onkeydown(viz.KEY_DOWN, orbit.down)
	vizact.onkeydown(viz.KEY_LEFT, orbit.left)
	vizact.onkeydown(viz.KEY_RIGHT, orbit.right)
	vizact.onkeydown('q', orbit.rollLeft)
	vizact.onkeydown('e', orbit.rollRight)
	
	vizact.onkeydown('1', orbit.setDistance, 1)
	vizact.onkeydown('2', orbit.setDistance, 5)
	vizact.onkeydown('3', orbit.setDistance, 10)
	
	base = viz.addGroup()
	box = viz.add("arrow.wrl")
	box.setPosition(0, 0, 1.5)
	box.setScale([0.15]*3)
	box.setParent(base)
	
	vizact.onkeydown('5', orbit.setNode, base)
	
	group = viz.addGroup()
	group.setParent(orbit)
	group.setEuler([0, 0, 0])
	link = viz.link(group, viz.MainView)
	link.setSrcFlag(viz.ABS_GLOBAL)
#	link = viz.link(orbit, viz.MainView)
